-- ============================================================
-- DBAOps Sentinel: License-Based Industry Compliance Access
-- Each license tier unlocks specific industry compliance modules
-- ============================================================

IF NOT EXISTS (SELECT 1 FROM sys.schemas WHERE name = 'licensing')
    EXEC('CREATE SCHEMA licensing');
GO

-- ============================================================
-- LICENSE TIERS
-- ============================================================
CREATE TABLE licensing.Tiers (
    TierID INT IDENTITY(1,1) PRIMARY KEY,
    TierCode NVARCHAR(20) NOT NULL UNIQUE,
    TierName NVARCHAR(50) NOT NULL,
    TierLevel INT NOT NULL, -- 0=Free, 1=Starter, 2=Pro, 3=Enterprise, 4=Ultimate
    Description NVARCHAR(500),
    
    -- Pricing
    MonthlyPrice DECIMAL(10,2),
    AnnualPrice DECIMAL(10,2),
    AnnualDiscount AS (CAST((1 - (AnnualPrice / NULLIF(MonthlyPrice * 12, 0))) * 100 AS INT)),
    
    -- Capacity Limits
    MaxServers INT,
    MaxDatabases INT,
    MaxUsers INT,
    DataRetentionDays INT,
    
    -- Core Features
    IncludesMonitoring BIT DEFAULT 1,
    IncludesAlerts BIT DEFAULT 1,
    IncludesSelfHealing BIT DEFAULT 0,
    IncludesHADR BIT DEFAULT 0,
    IncludesSecurityCenter BIT DEFAULT 0,
    IncludesAuditTrail BIT DEFAULT 0,
    IncludesReportFactory BIT DEFAULT 0,
    IncludesMobileApp BIT DEFAULT 0,
    IncludesAPIAccess BIT DEFAULT 0,
    IncludesAgentDeployment BIT DEFAULT 0,
    
    -- Support Level
    SupportLevel NVARCHAR(50), -- Community, Email, Priority, Premium, Dedicated
    SupportResponseHours INT,
    IncludesPhoneSupport BIT DEFAULT 0,
    IncludesDedicatedCSM BIT DEFAULT 0,
    
    -- Industry Compliance Access (JSON array of industry codes)
    IncludedIndustries NVARCHAR(MAX), -- ["PRIVATE"] or ["ALL"]
    MaxIndustryAddons INT DEFAULT 0,
    
    IsActive BIT DEFAULT 1,
    SortOrder INT DEFAULT 100,
    CreatedAt DATETIME2 DEFAULT GETDATE()
);

-- Pre-populate License Tiers
INSERT INTO licensing.Tiers (
    TierCode, TierName, TierLevel, Description,
    MonthlyPrice, AnnualPrice, 
    MaxServers, MaxDatabases, MaxUsers, DataRetentionDays,
    IncludesMonitoring, IncludesAlerts, IncludesSelfHealing, IncludesHADR,
    IncludesSecurityCenter, IncludesAuditTrail, IncludesReportFactory,
    IncludesMobileApp, IncludesAPIAccess, IncludesAgentDeployment,
    SupportLevel, SupportResponseHours, IncludesPhoneSupport, IncludesDedicatedCSM,
    IncludedIndustries, MaxIndustryAddons, SortOrder
)
VALUES
-- FREE TIER
('FREE', 'Free', 0, 
 'Essential monitoring for small environments. Perfect for evaluation.',
 0, 0,
 2, 5, 2, 7,
 1, 1, 0, 0, -- Monitoring, Alerts only
 0, 0, 0, -- No Security, Audit, Reports
 0, 0, 0, -- No Mobile, API, Agents
 'Community', NULL, 0, 0,
 '["PRIVATE"]', 0, 1), -- Only Private Sector (SOC 2)

-- STARTER TIER
('STARTER', 'Starter', 1,
 'Full monitoring for growing teams. Includes basic compliance.',
 29.99, 299.99,
 5, 20, 5, 30,
 1, 1, 0, 0, -- Monitoring, Alerts
 1, 0, 1, -- Security Center, Reports (no Audit)
 1, 0, 0, -- Mobile App
 'Email', 24, 0, 0,
 '["PRIVATE","REALESTATE"]', 1, 2), -- Private + Real Estate, can add 1 more

-- PROFESSIONAL TIER
('PROFESSIONAL', 'Professional', 2,
 'Complete operations suite with self-healing and industry compliance.',
 99.99, 999.99,
 15, 100, 15, 90,
 1, 1, 1, 1, -- All operations
 1, 1, 1, -- All security features
 1, 1, 1, -- Mobile, API, Agents
 'Priority', 8, 0, 0,
 '["PRIVATE","REALESTATE","EDUCATION","INSURANCE"]', 2, 3), -- 4 industries, can add 2 more

-- ENTERPRISE TIER  
('ENTERPRISE', 'Enterprise', 3,
 'Enterprise-grade compliance for regulated industries.',
 299.99, 2999.99,
 50, 500, 50, 365,
 1, 1, 1, 1,
 1, 1, 1,
 1, 1, 1,
 'Premium', 4, 1, 0,
 '["PRIVATE","REALESTATE","EDUCATION","INSURANCE","BANKING","HEALTHCARE"]', 4, 4), -- 6 industries, add 4

-- ULTIMATE TIER
('ULTIMATE', 'Ultimate', 4,
 'Unlimited everything. All industries. Dedicated support.',
 599.99, 5999.99,
 -1, -1, -1, 730, -- -1 = Unlimited
 1, 1, 1, 1,
 1, 1, 1,
 1, 1, 1,
 'Dedicated', 1, 1, 1,
 '["ALL"]', -1, 5), -- All industries included

-- GOVERNMENT TIER (Special)
('GOVERNMENT', 'Government', 5,
 'CJIS/FedRAMP compliant. For law enforcement and government agencies.',
 NULL, NULL, -- Contact Sales
 -1, -1, -1, 2555, -- 7 years retention for compliance
 1, 1, 1, 1,
 1, 1, 1,
 1, 1, 1,
 'Dedicated', 1, 1, 1,
 '["ALL"]', -1, 6);

-- ============================================================
-- INDUSTRY COMPLIANCE MODULES
-- ============================================================
CREATE TABLE licensing.IndustryModules (
    IndustryID INT IDENTITY(1,1) PRIMARY KEY,
    IndustryCode NVARCHAR(20) NOT NULL UNIQUE,
    IndustryName NVARCHAR(100) NOT NULL,
    Icon NVARCHAR(10),
    ComplianceFramework NVARCHAR(50),
    Description NVARCHAR(500),
    
    -- Pricing (for add-on purchase)
    MonthlyAddonPrice DECIMAL(10,2),
    AnnualAddonPrice DECIMAL(10,2),
    
    -- Minimum tier required to purchase as addon
    MinimumTierLevel INT DEFAULT 1,
    
    -- Compliance specifics
    TotalControls INT,
    RetentionRequirementDays INT,
    AuditFrequency NVARCHAR(50), -- Continuous, Daily, Weekly, Monthly, Quarterly, Annual
    CertificationRequired BIT DEFAULT 0,
    
    -- Features specific to this industry
    SpecificFeatures NVARCHAR(MAX), -- JSON array
    
    IsActive BIT DEFAULT 1,
    SortOrder INT DEFAULT 100
);

-- Pre-populate Industry Modules
INSERT INTO licensing.IndustryModules (
    IndustryCode, IndustryName, Icon, ComplianceFramework, Description,
    MonthlyAddonPrice, AnnualAddonPrice, MinimumTierLevel,
    TotalControls, RetentionRequirementDays, AuditFrequency, CertificationRequired,
    SpecificFeatures, SortOrder
)
VALUES
('LAWENFORCE', 'Law Enforcement', '🚔', 'CJIS 5.9.3',
 'Criminal Justice Information Services compliance for law enforcement agencies.',
 99.99, 999.99, 3, -- Enterprise minimum
 42, 365, 'Continuous', 1,
 '["CJI_Data_Encryption","Background_Checks","Advanced_Auth","Security_Awareness","Incident_Response","Media_Protection"]', 1),

('HEALTHCARE', 'Healthcare', '🏥', 'HIPAA',
 'Health Insurance Portability and Accountability Act compliance.',
 79.99, 799.99, 3, -- Enterprise minimum
 156, 2190, 'Continuous', 1,
 '["PHI_Protection","Access_Controls","Audit_Controls","Transmission_Security","Breach_Notification","BAA_Management"]', 2),

('BANKING', 'Banking', '🏦', 'PCI-DSS 4.0',
 'Payment Card Industry Data Security Standard for financial services.',
 79.99, 799.99, 3, -- Enterprise minimum
 120, 365, 'Quarterly', 1,
 '["Cardholder_Data_Protection","Network_Security","Vulnerability_Management","Access_Control","Monitoring_Testing","Security_Policy"]', 3),

('INSURANCE', 'Insurance', '🏢', 'SOX',
 'Sarbanes-Oxley Act compliance for insurance and financial services.',
 49.99, 499.99, 2, -- Professional minimum
 86, 2555, 'Annual', 1,
 '["Financial_Controls","Change_Management","Access_Reviews","Segregation_Duties","Audit_Trails","Data_Integrity"]', 4),

('EDUCATION', 'Education', '🎓', 'FERPA',
 'Family Educational Rights and Privacy Act for educational institutions.',
 39.99, 399.99, 2, -- Professional minimum
 45, 1825, 'Annual', 0,
 '["Student_Data_Protection","Consent_Management","Access_Controls","Directory_Info","Record_Amendments","Disclosure_Tracking"]', 5),

('GOVERNMENT', 'Government', '🏛️', 'FedRAMP/NIST',
 'Federal Risk and Authorization Management Program compliance.',
 149.99, 1499.99, 3, -- Enterprise minimum
 325, 2555, 'Continuous', 1,
 '["NIST_800_53","FedRAMP_Controls","Continuous_Monitoring","POA_M_Management","Authorization_Package","Security_Assessment"]', 6),

('REALESTATE', 'Real Estate', '🏠', 'GLBA',
 'Gramm-Leach-Bliley Act compliance for real estate and mortgage.',
 29.99, 299.99, 1, -- Starter minimum
 38, 1095, 'Annual', 0,
 '["NPI_Protection","Privacy_Notices","Safeguards_Rule","Pretexting_Protection","Information_Sharing","Risk_Assessment"]', 7),

('PRIVATE', 'Private Sector', '🏭', 'SOC 2',
 'Service Organization Control 2 for general business operations.',
 0, 0, 0, -- Included in Free
 64, 365, 'Annual', 0,
 '["Security_Principle","Availability_Principle","Processing_Integrity","Confidentiality","Privacy_Principle","Trust_Services"]', 8);

-- ============================================================
-- LICENSE INDUSTRY ACCESS (Junction Table)
-- ============================================================
CREATE TABLE licensing.TierIndustryAccess (
    AccessID INT IDENTITY(1,1) PRIMARY KEY,
    TierID INT FOREIGN KEY REFERENCES licensing.Tiers(TierID),
    IndustryID INT FOREIGN KEY REFERENCES licensing.IndustryModules(IndustryID),
    AccessType NVARCHAR(20) DEFAULT 'Included', -- Included, Addon, Locked
    UNIQUE(TierID, IndustryID)
);

-- Populate access matrix
-- FREE: Only Private Sector
INSERT INTO licensing.TierIndustryAccess (TierID, IndustryID, AccessType)
SELECT t.TierID, i.IndustryID,
    CASE 
        WHEN i.IndustryCode = 'PRIVATE' THEN 'Included'
        ELSE 'Locked'
    END
FROM licensing.Tiers t, licensing.IndustryModules i
WHERE t.TierCode = 'FREE';

-- STARTER: Private + Real Estate included, Education/Insurance as addon
INSERT INTO licensing.TierIndustryAccess (TierID, IndustryID, AccessType)
SELECT t.TierID, i.IndustryID,
    CASE 
        WHEN i.IndustryCode IN ('PRIVATE', 'REALESTATE') THEN 'Included'
        WHEN i.IndustryCode IN ('EDUCATION', 'INSURANCE') THEN 'Addon'
        ELSE 'Locked'
    END
FROM licensing.Tiers t, licensing.IndustryModules i
WHERE t.TierCode = 'STARTER';

-- PROFESSIONAL: 4 included, 2 as addon, 2 locked
INSERT INTO licensing.TierIndustryAccess (TierID, IndustryID, AccessType)
SELECT t.TierID, i.IndustryID,
    CASE 
        WHEN i.IndustryCode IN ('PRIVATE', 'REALESTATE', 'EDUCATION', 'INSURANCE') THEN 'Included'
        WHEN i.IndustryCode IN ('BANKING', 'HEALTHCARE') THEN 'Addon'
        ELSE 'Locked'
    END
FROM licensing.Tiers t, licensing.IndustryModules i
WHERE t.TierCode = 'PROFESSIONAL';

-- ENTERPRISE: 6 included, 2 as addon
INSERT INTO licensing.TierIndustryAccess (TierID, IndustryID, AccessType)
SELECT t.TierID, i.IndustryID,
    CASE 
        WHEN i.IndustryCode IN ('PRIVATE', 'REALESTATE', 'EDUCATION', 'INSURANCE', 'BANKING', 'HEALTHCARE') THEN 'Included'
        ELSE 'Addon'
    END
FROM licensing.Tiers t, licensing.IndustryModules i
WHERE t.TierCode = 'ENTERPRISE';

-- ULTIMATE & GOVERNMENT: All included
INSERT INTO licensing.TierIndustryAccess (TierID, IndustryID, AccessType)
SELECT t.TierID, i.IndustryID, 'Included'
FROM licensing.Tiers t, licensing.IndustryModules i
WHERE t.TierCode IN ('ULTIMATE', 'GOVERNMENT');

-- ============================================================
-- CUSTOMER LICENSES
-- ============================================================
CREATE TABLE licensing.CustomerLicenses (
    LicenseID INT IDENTITY(1,1) PRIMARY KEY,
    LicenseKey UNIQUEIDENTIFIER DEFAULT NEWID(),
    CustomerID INT,
    CustomerName NVARCHAR(200) NOT NULL,
    CustomerEmail NVARCHAR(200),
    CustomerPhone NVARCHAR(50),
    
    -- License Details
    TierID INT FOREIGN KEY REFERENCES licensing.Tiers(TierID),
    BillingCycle NVARCHAR(20), -- Monthly, Annual
    
    -- Dates
    PurchaseDate DATETIME2 DEFAULT GETDATE(),
    ActivatedDate DATETIME2,
    ExpirationDate DATETIME2,
    LastRenewalDate DATETIME2,
    
    -- Usage
    CurrentServers INT DEFAULT 0,
    CurrentDatabases INT DEFAULT 0,
    CurrentUsers INT DEFAULT 0,
    
    -- Add-on Industries
    AddonIndustries NVARCHAR(MAX), -- JSON array of purchased addon industry codes
    
    -- Status
    Status NVARCHAR(20) DEFAULT 'Active', -- Active, Trial, Expired, Suspended, Cancelled
    TrialDaysRemaining INT,
    
    -- Metadata
    Industry NVARCHAR(100),
    CompanySize NVARCHAR(50),
    Notes NVARCHAR(MAX),
    
    CreatedAt DATETIME2 DEFAULT GETDATE(),
    ModifiedAt DATETIME2 DEFAULT GETDATE()
);

-- ============================================================
-- ADDON PURCHASES
-- ============================================================
CREATE TABLE licensing.AddonPurchases (
    PurchaseID INT IDENTITY(1,1) PRIMARY KEY,
    LicenseID INT FOREIGN KEY REFERENCES licensing.CustomerLicenses(LicenseID),
    IndustryID INT FOREIGN KEY REFERENCES licensing.IndustryModules(IndustryID),
    PurchaseDate DATETIME2 DEFAULT GETDATE(),
    BillingCycle NVARCHAR(20),
    Price DECIMAL(10,2),
    Status NVARCHAR(20) DEFAULT 'Active',
    ExpirationDate DATETIME2
);

-- ============================================================
-- VIEW: Customer Industry Access
-- ============================================================
CREATE VIEW licensing.vw_CustomerIndustryAccess AS
SELECT 
    cl.LicenseID,
    cl.CustomerName,
    cl.LicenseKey,
    t.TierCode,
    t.TierName,
    im.IndustryCode,
    im.IndustryName,
    im.Icon,
    im.ComplianceFramework,
    CASE 
        WHEN tia.AccessType = 'Included' THEN 'Included'
        WHEN ap.PurchaseID IS NOT NULL AND ap.Status = 'Active' THEN 'Purchased'
        WHEN tia.AccessType = 'Addon' THEN 'Available for Purchase'
        ELSE 'Locked - Upgrade Required'
    END AS AccessStatus,
    CASE 
        WHEN tia.AccessType = 'Included' OR ap.PurchaseID IS NOT NULL THEN 1
        ELSE 0
    END AS HasAccess
FROM licensing.CustomerLicenses cl
JOIN licensing.Tiers t ON cl.TierID = t.TierID
CROSS JOIN licensing.IndustryModules im
LEFT JOIN licensing.TierIndustryAccess tia ON t.TierID = tia.TierID AND im.IndustryID = tia.IndustryID
LEFT JOIN licensing.AddonPurchases ap ON cl.LicenseID = ap.LicenseID AND im.IndustryID = ap.IndustryID AND ap.Status = 'Active'
WHERE cl.Status = 'Active';

PRINT '✅ License-Based Industry Compliance Schema Created';
PRINT '✅ 6 License Tiers Configured';
PRINT '✅ 8 Industry Compliance Modules Defined';
PRINT '✅ Access Matrix Populated';
