-- ============================================================
-- DBAOps Sentinel: License Management Schema
-- ============================================================

IF NOT EXISTS (SELECT 1 FROM sys.schemas WHERE name = 'licensing')
    EXEC('CREATE SCHEMA licensing');
GO

-- ============================================================
-- LICENSE KEYS TABLE
-- ============================================================
IF NOT EXISTS (SELECT * FROM sys.tables WHERE name = 'LicenseKeys' AND schema_id = SCHEMA_ID('licensing'))
CREATE TABLE licensing.LicenseKeys (
    LicenseID INT IDENTITY(1,1) PRIMARY KEY,
    LicenseKey NVARCHAR(30) NOT NULL UNIQUE,
    
    -- Tier Information
    TierCode NVARCHAR(20) NOT NULL,
    TierName NVARCHAR(50) NOT NULL,
    
    -- Customer Information
    CustomerID INT,
    CustomerName NVARCHAR(200),
    CustomerEmail NVARCHAR(200),
    CustomerPhone NVARCHAR(50),
    CompanyName NVARCHAR(200),
    
    -- License Details
    MaxServers INT DEFAULT 5,
    MaxDatabases INT DEFAULT 20,
    MaxUsers INT DEFAULT 5,
    
    -- Features Enabled (JSON)
    FeaturesEnabled NVARCHAR(MAX),
    IndustriesEnabled NVARCHAR(MAX),
    
    -- Validity
    GeneratedAt DATETIME2 DEFAULT GETDATE(),
    ActivatedAt DATETIME2,
    ExpiresAt DATETIME2,
    
    -- Status
    Status NVARCHAR(20) DEFAULT 'Pending', -- Pending, Active, Expired, Revoked, Suspended
    
    -- Activation Details
    ActivationCount INT DEFAULT 0,
    MaxActivations INT DEFAULT 3,
    LastActivationIP NVARCHAR(50),
    LastActivationMachine NVARCHAR(200),
    
    -- Metadata
    Notes NVARCHAR(MAX),
    CreatedBy NVARCHAR(100),
    ModifiedAt DATETIME2 DEFAULT GETDATE()
);

-- Index for fast lookups
CREATE INDEX IX_LicenseKeys_Key ON licensing.LicenseKeys(LicenseKey);
CREATE INDEX IX_LicenseKeys_Customer ON licensing.LicenseKeys(CustomerEmail);
CREATE INDEX IX_LicenseKeys_Status ON licensing.LicenseKeys(Status);

-- ============================================================
-- LICENSE ACTIVATIONS TABLE
-- ============================================================
IF NOT EXISTS (SELECT * FROM sys.tables WHERE name = 'LicenseActivations' AND schema_id = SCHEMA_ID('licensing'))
CREATE TABLE licensing.LicenseActivations (
    ActivationID INT IDENTITY(1,1) PRIMARY KEY,
    LicenseID INT FOREIGN KEY REFERENCES licensing.LicenseKeys(LicenseID),
    LicenseKey NVARCHAR(30),
    
    -- Machine Information
    MachineID NVARCHAR(100),
    MachineName NVARCHAR(200),
    MachineIP NVARCHAR(50),
    OperatingSystem NVARCHAR(100),
    
    -- Activation Details
    ActivatedAt DATETIME2 DEFAULT GETDATE(),
    DeactivatedAt DATETIME2,
    IsActive BIT DEFAULT 1,
    
    -- Usage Tracking
    LastHeartbeat DATETIME2,
    ServerCount INT DEFAULT 0,
    DatabaseCount INT DEFAULT 0
);

-- ============================================================
-- STORED PROCEDURES
-- ============================================================

-- Procedure: Generate New License Key
CREATE OR ALTER PROCEDURE licensing.GenerateLicenseKey
    @TierCode NVARCHAR(20),
    @CustomerName NVARCHAR(200),
    @CustomerEmail NVARCHAR(200),
    @ValidDays INT = 365,
    @CreatedBy NVARCHAR(100) = 'System',
    @LicenseKey NVARCHAR(30) OUTPUT
AS
BEGIN
    SET NOCOUNT ON;
    
    -- Generate unique key: SENT-XXXX-XXXX-XXXX-XXXX
    DECLARE @TierPrefix CHAR(1) = LEFT(@TierCode, 1);
    DECLARE @Random1 NVARCHAR(3) = (SELECT TOP 3 CHAR(65 + ABS(CHECKSUM(NEWID())) % 26) FROM sys.objects);
    DECLARE @Segment2 NVARCHAR(4) = SUBSTRING(REPLACE(CONVERT(NVARCHAR(36), NEWID()), '-', ''), 1, 4);
    DECLARE @Segment3 NVARCHAR(4) = SUBSTRING(REPLACE(CONVERT(NVARCHAR(36), NEWID()), '-', ''), 1, 4);
    DECLARE @Segment4 NVARCHAR(4) = SUBSTRING(REPLACE(CONVERT(NVARCHAR(36), NEWID()), '-', ''), 1, 4);
    
    SET @LicenseKey = CONCAT('SENT-', @TierPrefix, @Random1, '-', UPPER(@Segment2), '-', UPPER(@Segment3), '-', UPPER(@Segment4));
    
    -- Get tier details
    DECLARE @TierName NVARCHAR(50), @MaxServers INT, @MaxDatabases INT;
    SELECT @TierName = TierName, @MaxServers = MaxServers, @MaxDatabases = MaxDatabases
    FROM licensing.Tiers WHERE TierCode = @TierCode;
    
    -- Insert license
    INSERT INTO licensing.LicenseKeys (
        LicenseKey, TierCode, TierName, CustomerName, CustomerEmail,
        MaxServers, MaxDatabases, ExpiresAt, Status, CreatedBy
    ) VALUES (
        @LicenseKey, @TierCode, @TierName, @CustomerName, @CustomerEmail,
        @MaxServers, @MaxDatabases, DATEADD(DAY, @ValidDays, GETDATE()), 'Pending', @CreatedBy
    );
    
    SELECT @LicenseKey AS GeneratedKey;
END;
GO

-- Procedure: Validate and Activate License
CREATE OR ALTER PROCEDURE licensing.ActivateLicense
    @LicenseKey NVARCHAR(30),
    @MachineID NVARCHAR(100),
    @MachineName NVARCHAR(200),
    @MachineIP NVARCHAR(50),
    @Success BIT OUTPUT,
    @Message NVARCHAR(500) OUTPUT,
    @TierCode NVARCHAR(20) OUTPUT,
    @ExpiresAt DATETIME2 OUTPUT
AS
BEGIN
    SET NOCOUNT ON;
    SET @Success = 0;
    
    -- Check if license exists
    IF NOT EXISTS (SELECT 1 FROM licensing.LicenseKeys WHERE LicenseKey = @LicenseKey)
    BEGIN
        SET @Message = 'Invalid license key';
        RETURN;
    END
    
    DECLARE @LicenseID INT, @Status NVARCHAR(20), @ActivationCount INT, @MaxActivations INT, @ExpDate DATETIME2;
    
    SELECT @LicenseID = LicenseID, @Status = Status, @TierCode = TierCode,
           @ActivationCount = ActivationCount, @MaxActivations = MaxActivations, @ExpDate = ExpiresAt
    FROM licensing.LicenseKeys WHERE LicenseKey = @LicenseKey;
    
    -- Check status
    IF @Status = 'Revoked'
    BEGIN
        SET @Message = 'This license has been revoked';
        RETURN;
    END
    
    IF @Status = 'Suspended'
    BEGIN
        SET @Message = 'This license is suspended. Contact support.';
        RETURN;
    END
    
    -- Check expiration
    IF @ExpDate < GETDATE()
    BEGIN
        UPDATE licensing.LicenseKeys SET Status = 'Expired' WHERE LicenseID = @LicenseID;
        SET @Message = 'This license has expired';
        RETURN;
    END
    
    -- Check activation limit
    IF @ActivationCount >= @MaxActivations
    BEGIN
        -- Check if this machine is already activated
        IF NOT EXISTS (SELECT 1 FROM licensing.LicenseActivations WHERE LicenseID = @LicenseID AND MachineID = @MachineID AND IsActive = 1)
        BEGIN
            SET @Message = 'Maximum activation limit reached. Deactivate another machine first.';
            RETURN;
        END
    END
    
    -- Activate or update activation
    IF EXISTS (SELECT 1 FROM licensing.LicenseActivations WHERE LicenseID = @LicenseID AND MachineID = @MachineID)
    BEGIN
        UPDATE licensing.LicenseActivations 
        SET IsActive = 1, LastHeartbeat = GETDATE(), MachineIP = @MachineIP, DeactivatedAt = NULL
        WHERE LicenseID = @LicenseID AND MachineID = @MachineID;
    END
    ELSE
    BEGIN
        INSERT INTO licensing.LicenseActivations (LicenseID, LicenseKey, MachineID, MachineName, MachineIP)
        VALUES (@LicenseID, @LicenseKey, @MachineID, @MachineName, @MachineIP);
        
        UPDATE licensing.LicenseKeys 
        SET ActivationCount = ActivationCount + 1, ActivatedAt = COALESCE(ActivatedAt, GETDATE()),
            Status = 'Active', LastActivationIP = @MachineIP, LastActivationMachine = @MachineName
        WHERE LicenseID = @LicenseID;
    END
    
    SET @Success = 1;
    SET @Message = 'License activated successfully';
    SET @ExpiresAt = @ExpDate;
END;
GO

-- Procedure: Deactivate License
CREATE OR ALTER PROCEDURE licensing.DeactivateLicense
    @LicenseKey NVARCHAR(30),
    @MachineID NVARCHAR(100)
AS
BEGIN
    SET NOCOUNT ON;
    
    UPDATE licensing.LicenseActivations 
    SET IsActive = 0, DeactivatedAt = GETDATE()
    WHERE LicenseKey = @LicenseKey AND MachineID = @MachineID;
    
    UPDATE licensing.LicenseKeys 
    SET ActivationCount = ActivationCount - 1
    WHERE LicenseKey = @LicenseKey AND ActivationCount > 0;
END;
GO

-- Procedure: Check License Status
CREATE OR ALTER PROCEDURE licensing.CheckLicenseStatus
    @LicenseKey NVARCHAR(30)
AS
BEGIN
    SET NOCOUNT ON;
    
    SELECT 
        k.LicenseKey,
        k.TierCode,
        k.TierName,
        k.CustomerName,
        k.Status,
        k.MaxServers,
        k.MaxDatabases,
        k.ExpiresAt,
        k.ActivationCount,
        k.MaxActivations,
        CASE WHEN k.ExpiresAt > GETDATE() THEN 1 ELSE 0 END AS IsValid,
        DATEDIFF(DAY, GETDATE(), k.ExpiresAt) AS DaysRemaining
    FROM licensing.LicenseKeys k
    WHERE k.LicenseKey = @LicenseKey;
    
    -- Return active installations
    SELECT 
        a.MachineName,
        a.MachineIP,
        a.ActivatedAt,
        a.LastHeartbeat,
        a.ServerCount,
        a.DatabaseCount
    FROM licensing.LicenseActivations a
    WHERE a.LicenseKey = @LicenseKey AND a.IsActive = 1;
END;
GO

PRINT '✅ License Management Schema Created';
