-- ============================================================
-- DBAOps: Enterprise Connectors Schema
-- ============================================================
IF NOT EXISTS (SELECT 1 FROM sys.schemas WHERE name = 'connector')
    EXEC('CREATE SCHEMA connector');
GO

-- Connector Definitions
CREATE TABLE connector.Definitions (
    ConnectorID INT IDENTITY(1,1) PRIMARY KEY,
    ConnectorName NVARCHAR(100) NOT NULL,
    ConnectorType NVARCHAR(50) NOT NULL,
    -- ActiveDirectory, EntraID, VMware, Azure, AWS, GCP, ServiceNow, 
    -- SQLServer, DNS, DHCP, Kubernetes, Tanium, Intune, CyberArk
    Description NVARCHAR(500),
    IconURL NVARCHAR(500),
    
    -- Connection
    ConnectionConfig NVARCHAR(MAX), -- Encrypted JSON
    CredentialID INT, -- FK to secrets
    
    -- Permissions
    RequiredPermissions NVARCHAR(MAX), -- JSON array
    PermissionLevel NVARCHAR(20) DEFAULT 'ReadOnly', -- ReadOnly, ReadWrite, Admin
    
    -- Scheduling
    IsEnabled BIT DEFAULT 0,
    SyncIntervalMinutes INT DEFAULT 60,
    LastSync DATETIME2,
    NextSync DATETIME2,
    SyncStatus NVARCHAR(20),
    
    -- Scope
    ScopeFilter NVARCHAR(MAX), -- JSON filter criteria
    
    CreatedAt DATETIME2 DEFAULT GETDATE(),
    CreatedBy NVARCHAR(128)
);

-- Pre-populate connector types
INSERT INTO connector.Definitions (ConnectorName, ConnectorType, Description, RequiredPermissions, SyncIntervalMinutes)
VALUES
('Active Directory', 'ActiveDirectory', 'Discover servers and service accounts from AD', 
 '["Read all properties","Read servicePrincipalName"]', 240),
('Microsoft Entra ID', 'EntraID', 'Azure AD / Entra ID integration for users and groups',
 '["Directory.Read.All","User.Read.All"]', 360),
('VMware vCenter', 'VMware', 'Discover VMs running SQL Server',
 '["System.View","VirtualMachine.Inventory"]', 120),
('Azure Resource Manager', 'Azure', 'Discover Azure SQL and SQL VMs',
 '["Reader on Subscription","SQL DB Contributor"]', 60),
('AWS RDS', 'AWS', 'Discover RDS SQL Server instances',
 '["rds:DescribeDBInstances","ec2:DescribeInstances"]', 60),
('ServiceNow CMDB', 'ServiceNow', 'Sync with ServiceNow CMDB',
 '["cmdb_read","cmdb_ci_server"]', 360),
('SQL Server CMS', 'SQLServer', 'Central Management Server groups',
 '["VIEW SERVER STATE"]', 30),
('Kubernetes', 'Kubernetes', 'Discover SQL containers in K8s',
 '["pods:list","services:list"]', 60);

-- Connector Sync Log
CREATE TABLE connector.SyncLog (
    SyncID BIGINT IDENTITY(1,1) PRIMARY KEY,
    ConnectorID INT FOREIGN KEY REFERENCES connector.Definitions(ConnectorID),
    StartTime DATETIME2 DEFAULT GETDATE(),
    EndTime DATETIME2,
    Status NVARCHAR(20), -- Running, Completed, Failed, Cancelled
    AssetsDiscovered INT DEFAULT 0,
    AssetsCreated INT DEFAULT 0,
    AssetsUpdated INT DEFAULT 0,
    AssetsRemoved INT DEFAULT 0,
    ErrorCount INT DEFAULT 0,
    ErrorDetails NVARCHAR(MAX),
    INDEX IX_Sync_Time NONCLUSTERED (StartTime DESC)
);

PRINT '✅ Enterprise Connectors Schema Created';
