-- ============================================================
-- DBAOps: Telemetry Pipeline & Diagnostics Schema
-- ============================================================

-- Telemetry Queue (offline buffering)
IF NOT EXISTS (SELECT 1 FROM sys.schemas WHERE name = 'telemetry')
    EXEC('CREATE SCHEMA telemetry');
GO

CREATE TABLE telemetry.MessageQueue (
    MessageID BIGINT IDENTITY(1,1) PRIMARY KEY,
    QueuedAt DATETIME2 DEFAULT GETDATE(),
    MessageType NVARCHAR(50), -- Metric, Event, Log, Alert
    Priority INT DEFAULT 5, -- 1 = highest
    PayloadJSON NVARCHAR(MAX),
    SourceAgentID UNIQUEIDENTIFIER,
    SourceHost NVARCHAR(256),
    DeliveryAttempts INT DEFAULT 0,
    LastAttempt DATETIME2,
    Status NVARCHAR(20) DEFAULT 'Pending', -- Pending, Processing, Delivered, Failed, Expired
    ExpiresAt DATETIME2,
    IdempotencyKey NVARCHAR(100), -- For exactly-once semantics
    INDEX IX_Queue_Status NONCLUSTERED (Status, Priority, QueuedAt)
);

-- Delivered Messages (for idempotency check)
CREATE TABLE telemetry.DeliveredMessages (
    IdempotencyKey NVARCHAR(100) PRIMARY KEY,
    DeliveredAt DATETIME2 DEFAULT GETDATE(),
    INDEX IX_Delivered_Time NONCLUSTERED (DeliveredAt)
);

-- SIEM Export Configuration
CREATE TABLE telemetry.SIEMExportConfig (
    ConfigID INT IDENTITY(1,1) PRIMARY KEY,
    SIEMType NVARCHAR(50), -- Splunk, Sentinel, QRadar, Syslog, Webhook
    IsEnabled BIT DEFAULT 0,
    
    -- Connection
    EndpointURL NVARCHAR(500),
    Port INT,
    Protocol NVARCHAR(20), -- TCP, UDP, HTTPS
    Format NVARCHAR(20), -- CEF, JSON, Syslog, Custom
    
    -- Filtering
    EventTypes NVARCHAR(MAX), -- JSON array of event types to export
    MinimumSeverity NVARCHAR(20),
    
    -- Authentication
    AuthType NVARCHAR(50),
    CredentialID INT,
    
    -- Status
    LastExport DATETIME2,
    ExportStatus NVARCHAR(20),
    
    CreatedAt DATETIME2 DEFAULT GETDATE()
);

-- Diagnostics Package
IF NOT EXISTS (SELECT 1 FROM sys.schemas WHERE name = 'diag')
    EXEC('CREATE SCHEMA diag');
GO

CREATE TABLE diag.DiagnosticRuns (
    RunID INT IDENTITY(1,1) PRIMARY KEY,
    RunType NVARCHAR(50), -- FullDiagnostic, ConnectivityTest, PermissionCheck, CoverageReport
    StartTime DATETIME2 DEFAULT GETDATE(),
    EndTime DATETIME2,
    TriggeredBy NVARCHAR(128),
    Status NVARCHAR(20), -- Running, Completed, Failed
    
    -- Results Summary
    TotalChecks INT,
    PassedChecks INT,
    FailedChecks INT,
    WarningChecks INT,
    
    -- Output
    ResultsJSON NVARCHAR(MAX),
    RecommendationsJSON NVARCHAR(MAX),
    
    -- Export
    ExportPath NVARCHAR(500),
    ExportFormat NVARCHAR(20)
);

-- Permission Gap Analysis
CREATE TABLE diag.PermissionGaps (
    GapID INT IDENTITY(1,1) PRIMARY KEY,
    DetectedAt DATETIME2 DEFAULT GETDATE(),
    TargetType NVARCHAR(50), -- Server, Database, AD, Azure, AWS
    TargetName NVARCHAR(256),
    RequiredPermission NVARCHAR(200),
    CurrentPermission NVARCHAR(200),
    GapDescription NVARCHAR(500),
    RemediationScript NVARCHAR(MAX),
    Severity NVARCHAR(20),
    Status NVARCHAR(20) DEFAULT 'Open' -- Open, Remediated, Acknowledged
);

-- Coverage Report
CREATE TABLE diag.CoverageReport (
    ReportID INT IDENTITY(1,1) PRIMARY KEY,
    ReportDate DATETIME2 DEFAULT GETDATE(),
    ScopeType NVARCHAR(50), -- OU, Subnet, Subscription, VCenter
    ScopeName NVARCHAR(256),
    ExpectedAssets INT,
    DiscoveredAssets INT,
    ManagedAssets INT,
    CoveragePercent AS (CAST(DiscoveredAssets AS DECIMAL(5,2)) / NULLIF(ExpectedAssets, 0) * 100),
    MissingAssets NVARCHAR(MAX), -- JSON array
    Notes NVARCHAR(MAX)
);

PRINT '✅ Telemetry Pipeline & Diagnostics Schema Created';
