-- ============================================================
-- DBAOps: Secure Secret Handling Schema
-- ============================================================
IF NOT EXISTS (SELECT 1 FROM sys.schemas WHERE name = 'secrets')
    EXEC('CREATE SCHEMA secrets');
GO

-- Secret Store Providers
CREATE TABLE secrets.SecretStoreProviders (
    ProviderID INT IDENTITY(1,1) PRIMARY KEY,
    ProviderName NVARCHAR(100) NOT NULL,
    ProviderType NVARCHAR(50) NOT NULL, -- WindowsDPAPI, AzureKeyVault, HashiCorpVault, AWSSecretsManager, CyberArk
    IsEnabled BIT DEFAULT 1,
    IsPrimary BIT DEFAULT 0,
    
    -- Connection Details (encrypted)
    ConnectionConfig NVARCHAR(MAX), -- Encrypted JSON
    
    -- Azure Key Vault
    AzureVaultURL NVARCHAR(500),
    AzureTenantID NVARCHAR(100),
    AzureClientID NVARCHAR(100),
    
    -- HashiCorp Vault
    VaultAddress NVARCHAR(500),
    VaultNamespace NVARCHAR(100),
    VaultAuthMethod NVARCHAR(50), -- Token, AppRole, Kubernetes, LDAP
    
    -- AWS Secrets Manager
    AWSRegion NVARCHAR(50),
    AWSRoleARN NVARCHAR(256),
    
    -- Status
    LastHealthCheck DATETIME2,
    HealthStatus NVARCHAR(20),
    
    CreatedAt DATETIME2 DEFAULT GETDATE()
);

-- Pre-populate providers
INSERT INTO secrets.SecretStoreProviders (ProviderName, ProviderType, IsPrimary)
VALUES
('Windows DPAPI', 'WindowsDPAPI', 1),
('Azure Key Vault', 'AzureKeyVault', 0),
('HashiCorp Vault', 'HashiCorpVault', 0),
('AWS Secrets Manager', 'AWSSecretsManager', 0);

-- Secret References (metadata only, not actual secrets)
CREATE TABLE secrets.SecretReferences (
    SecretID INT IDENTITY(1,1) PRIMARY KEY,
    SecretName NVARCHAR(200) NOT NULL,
    SecretType NVARCHAR(50), -- SQLCredential, ServiceAccount, APIKey, Certificate, ConnectionString
    ProviderID INT FOREIGN KEY REFERENCES secrets.SecretStoreProviders(ProviderID),
    
    -- Reference to external secret
    ExternalSecretID NVARCHAR(500), -- Key Vault secret name, Vault path, etc.
    ExternalVersion NVARCHAR(50),
    
    -- Metadata
    Description NVARCHAR(500),
    AssociatedServers NVARCHAR(MAX), -- JSON array of server names
    AssociatedConnectors NVARCHAR(MAX), -- JSON array of connector names
    
    -- Rotation
    RotationEnabled BIT DEFAULT 0,
    RotationIntervalDays INT,
    LastRotated DATETIME2,
    NextRotation DATETIME2,
    RotationStatus NVARCHAR(20),
    
    -- Audit
    LastAccessed DATETIME2,
    AccessCount INT DEFAULT 0,
    
    CreatedAt DATETIME2 DEFAULT GETDATE(),
    CreatedBy NVARCHAR(128),
    ModifiedAt DATETIME2,
    ModifiedBy NVARCHAR(128)
);

-- Secret Access Log (audit who accessed what)
CREATE TABLE secrets.SecretAccessLog (
    AccessID BIGINT IDENTITY(1,1) PRIMARY KEY,
    SecretID INT FOREIGN KEY REFERENCES secrets.SecretReferences(SecretID),
    AccessTimestamp DATETIME2 DEFAULT GETDATE(),
    AccessedBy NVARCHAR(128),
    AccessType NVARCHAR(20), -- Read, Write, Rotate, Delete
    AccessSource NVARCHAR(100), -- Agent, API, Console, Connector
    SourceIPAddress NVARCHAR(45),
    AccessResult NVARCHAR(20), -- Success, Denied, Error
    INDEX IX_SecretAccess_Time NONCLUSTERED (AccessTimestamp DESC)
);

-- Credential Redaction Patterns
CREATE TABLE secrets.RedactionPatterns (
    PatternID INT IDENTITY(1,1) PRIMARY KEY,
    PatternName NVARCHAR(100),
    PatternRegex NVARCHAR(500),
    ReplacementText NVARCHAR(50) DEFAULT '***REDACTED***',
    IsEnabled BIT DEFAULT 1
);

INSERT INTO secrets.RedactionPatterns (PatternName, PatternRegex, ReplacementText)
VALUES
('Password in Connection String', 'Password\s*=\s*[^;]+', 'Password=***'),
('PWD in Connection String', 'Pwd\s*=\s*[^;]+', 'Pwd=***'),
('API Key Header', 'api[_-]?key["\s:=]+[a-zA-Z0-9_-]+', 'api_key=***'),
('Bearer Token', 'Bearer\s+[a-zA-Z0-9_.-]+', 'Bearer ***'),
('AWS Secret Key', 'aws[_-]?secret[_-]?access[_-]?key["\s:=]+[^\s"]+', 'aws_secret_access_key=***'),
('Azure Client Secret', 'client[_-]?secret["\s:=]+[^\s"]+', 'client_secret=***');

PRINT '✅ Secure Secret Handling Schema Created';
