-- ============================================================
-- DBAOps: Enterprise Identity & Authentication Schema
-- ============================================================
IF NOT EXISTS (SELECT 1 FROM sys.schemas WHERE name = 'identity')
    EXEC('CREATE SCHEMA identity');
GO

-- Identity Providers
CREATE TABLE identity.IdentityProviders (
    ProviderID INT IDENTITY(1,1) PRIMARY KEY,
    ProviderName NVARCHAR(100) NOT NULL,
    ProviderType NVARCHAR(50) NOT NULL, -- ActiveDirectory, EntraID, SAML, OIDC, LDAP, Local
    IsEnabled BIT DEFAULT 1,
    IsPrimary BIT DEFAULT 0,
    Priority INT DEFAULT 100,
    
    -- AD/LDAP Settings
    DomainName NVARCHAR(256),
    DomainControllers NVARCHAR(MAX), -- JSON array
    BaseDN NVARCHAR(500),
    BindMethod NVARCHAR(50), -- Kerberos, NTLM, Simple
    UseTLS BIT DEFAULT 1,
    
    -- Entra ID / OIDC Settings
    TenantID NVARCHAR(100),
    ClientID NVARCHAR(100),
    Authority NVARCHAR(500),
    RedirectURI NVARCHAR(500),
    
    -- SAML Settings
    MetadataURL NVARCHAR(500),
    EntityID NVARCHAR(256),
    AssertionConsumerServiceURL NVARCHAR(500),
    
    -- Status
    LastSync DATETIME2,
    SyncStatus NVARCHAR(20),
    Configuration NVARCHAR(MAX), -- Encrypted JSON
    CreatedAt DATETIME2 DEFAULT GETDATE()
);

-- Pre-populate common providers
INSERT INTO identity.IdentityProviders (ProviderName, ProviderType, IsPrimary, Priority)
VALUES
('Local Database', 'Local', 1, 1),
('Active Directory', 'ActiveDirectory', 0, 10),
('Microsoft Entra ID', 'EntraID', 0, 20),
('SAML SSO', 'SAML', 0, 30),
('OIDC Provider', 'OIDC', 0, 40);

-- Service Accounts (gMSA tracking)
CREATE TABLE identity.ServiceAccounts (
    AccountID INT IDENTITY(1,1) PRIMARY KEY,
    AccountName NVARCHAR(200) NOT NULL,
    AccountType NVARCHAR(50) NOT NULL, -- gMSA, ManagedIdentity, ServicePrincipal, StandardService
    Domain NVARCHAR(256),
    SamAccountName NVARCHAR(128),
    UPN NVARCHAR(256),
    
    -- gMSA specific
    IsgMSA BIT DEFAULT 0,
    gMSADNSHostName NVARCHAR(256),
    gMSAManagedPasswordInterval INT,
    gMSAPrincipalsAllowed NVARCHAR(MAX),
    
    -- Azure Service Principal
    AzureClientID NVARCHAR(100),
    AzureTenantID NVARCHAR(100),
    AzureObjectID NVARCHAR(100),
    
    -- AWS IAM Role
    AWSRoleARN NVARCHAR(256),
    
    -- Certificate Auth
    CertificateThumbprint NVARCHAR(100),
    CertificateSubject NVARCHAR(256),
    CertificateExpiry DATETIME2,
    
    -- Usage
    AssociatedServices NVARCHAR(MAX), -- JSON array of services using this account
    Permissions NVARCHAR(MAX), -- JSON of permissions
    LastUsed DATETIME2,
    Status NVARCHAR(20) DEFAULT 'Active',
    ComplianceStatus NVARCHAR(20), -- Compliant, NonCompliant, Unknown
    CreatedAt DATETIME2 DEFAULT GETDATE()
);

-- RBAC Roles
CREATE TABLE identity.Roles (
    RoleID INT IDENTITY(1,1) PRIMARY KEY,
    RoleName NVARCHAR(50) NOT NULL UNIQUE,
    RoleType NVARCHAR(20) NOT NULL, -- BuiltIn, Custom
    Description NVARCHAR(500),
    Permissions NVARCHAR(MAX), -- JSON array of permissions
    IsDefault BIT DEFAULT 0,
    IsActive BIT DEFAULT 1,
    CreatedAt DATETIME2 DEFAULT GETDATE(),
    CreatedBy NVARCHAR(128)
);

-- Pre-populate RBAC roles
INSERT INTO identity.Roles (RoleName, RoleType, Description, Permissions, IsDefault)
VALUES
('Viewer', 'BuiltIn', 'Read-only access to dashboards and reports', 
 '["dashboard:read","reports:read","servers:read","alerts:read"]', 0),
('Operator', 'BuiltIn', 'Can acknowledge alerts and run basic actions', 
 '["dashboard:read","reports:read","servers:read","alerts:read","alerts:acknowledge","actions:basic"]', 0),
('DBA', 'BuiltIn', 'Full DBA operations access', 
 '["dashboard:*","reports:*","servers:*","alerts:*","actions:*","backup:*","maintenance:*"]', 1),
('SecurityAdmin', 'BuiltIn', 'Security and compliance management', 
 '["security:*","compliance:*","audit:*","users:read"]', 0),
('Admin', 'BuiltIn', 'Full administrative access', 
 '["*"]', 0);

-- User-Role Assignments
CREATE TABLE identity.UserRoleAssignments (
    AssignmentID INT IDENTITY(1,1) PRIMARY KEY,
    UserPrincipal NVARCHAR(256) NOT NULL, -- UPN or SamAccountName
    UserType NVARCHAR(50), -- User, Group, ServicePrincipal
    RoleID INT FOREIGN KEY REFERENCES identity.Roles(RoleID),
    Scope NVARCHAR(500), -- NULL = global, or specific scope like "server:PROD*"
    ScopeType NVARCHAR(50), -- Global, Server, Database, Environment
    AssignedAt DATETIME2 DEFAULT GETDATE(),
    AssignedBy NVARCHAR(128),
    ExpiresAt DATETIME2,
    IsActive BIT DEFAULT 1
);

-- Authentication Sessions
CREATE TABLE identity.Sessions (
    SessionID UNIQUEIDENTIFIER PRIMARY KEY DEFAULT NEWID(),
    UserPrincipal NVARCHAR(256) NOT NULL,
    IdentityProviderID INT,
    ClientIPAddress NVARCHAR(45),
    ClientUserAgent NVARCHAR(500),
    DeviceID NVARCHAR(100),
    CreatedAt DATETIME2 DEFAULT GETDATE(),
    ExpiresAt DATETIME2,
    LastActivity DATETIME2,
    IsActive BIT DEFAULT 1,
    MFAVerified BIT DEFAULT 0,
    AuthMethod NVARCHAR(50)
);

PRINT '✅ Enterprise Identity & Authentication Schema Created';
