<#
.SYNOPSIS
    Activate DBAOps Sentinel license
.EXAMPLE
    .\Activate-License.ps1 -Key "SENT-XXXX-XXXX-XXXX-XXXX"
    .\Activate-License.ps1 -Trial
    .\Activate-License.ps1 -Status
#>

param(
    [string]$Key,
    [switch]$Trial,
    [switch]$Status,
    [switch]$Deactivate
)

$configPath = $env:SENTINEL_CONFIG
if (-not $configPath -or !(Test-Path $configPath)) {
    $configPath = "C:\Program Files\DBAOps\Sentinel\Config\sentinel-config.json"
}

if (!(Test-Path $configPath)) {
    Write-Host "  Sentinel not installed. Run Install-Sentinel.ps1 first." -ForegroundColor Red
    exit 1
}

$config = Get-Content $configPath | ConvertFrom-Json

Write-Host ""
Write-Host "  DBAOps Sentinel - License Manager" -ForegroundColor Cyan
Write-Host ""

if ($Status) {
    Write-Host "  License Status:" -ForegroundColor Yellow
    Write-Host "  ─────────────────────────────────────" -ForegroundColor Gray
    Write-Host "  Type:      $($config.License.Type)"
    Write-Host "  Key:       $(if ($config.License.Key) { $config.License.Key } else { 'N/A' })"
    Write-Host "  Activated: $($config.License.Activated)"
    Write-Host "  Expires:   $(if ($config.License.ExpiresAt -eq '9999-12-31') { 'NEVER (Perpetual)' } elseif ($config.License.ExpiresAt) { $config.License.ExpiresAt } else { 'N/A' })"
    
    if ($config.License.ExpiresAt -and $config.License.ExpiresAt -ne '9999-12-31') {
        $days = ([DateTime]$config.License.ExpiresAt - (Get-Date)).Days
        if ($days -lt 0) { Write-Host "  Status:    EXPIRED" -ForegroundColor Red }
        elseif ($days -lt 30) { Write-Host "  Status:    Expires in $days days" -ForegroundColor Yellow }
        else { Write-Host "  Status:    Active ($days days)" -ForegroundColor Green }
    } else {
        Write-Host "  Status:    Active" -ForegroundColor Green
    }
    Write-Host ""
    exit 0
}

if ($Deactivate) {
    $config.License.Key = ""
    $config.License.Type = "Scout"
    $config.License.Activated = $false
    $config.License.ExpiresAt = $null
    $config | ConvertTo-Json -Depth 10 | Out-File $configPath -Encoding UTF8 -Force
    Write-Host "  License deactivated. Reverted to Scout (Free)." -ForegroundColor Green
    Write-Host ""
    exit 0
}

if ($Trial) {
    $config.License.Key = ""
    $config.License.Type = "Trial"
    $config.License.Activated = $true
    $config.License.ExpiresAt = (Get-Date).AddDays(90).ToString("yyyy-MM-dd")
    $config | ConvertTo-Json -Depth 10 | Out-File $configPath -Encoding UTF8 -Force
    Write-Host "  90-day trial activated!" -ForegroundColor Green
    Write-Host "  Expires: $($config.License.ExpiresAt)" -ForegroundColor Cyan
    Write-Host "  Features: Guard tier (Self-Healing, Compliance)" -ForegroundColor Cyan
    Write-Host ""
    exit 0
}

if ($Key) {
    if ($Key -notmatch "^SENT-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}$") {
        Write-Host "  Invalid license key format." -ForegroundColor Red
        Write-Host "  Expected: SENT-XXXX-XXXX-XXXX-XXXX" -ForegroundColor Yellow
        exit 1
    }
    
    $tierCode = $Key.Substring(5, 1)
    $tier = switch ($tierCode) {
        'W' { "Watch" }
        'G' { "Guard" }
        'C' { "Command" }
        'F' { "Fortress" }
        'P' { "Government" }
        'V' { "Government" }
        default { "Watch" }
    }
    
    $config.License.Key = $Key
    $config.License.Type = $tier
    $config.License.Activated = $true
    
    if ($tierCode -eq 'P') {
        $config.License.ExpiresAt = "9999-12-31"
        Write-Host "  PERPETUAL license activated!" -ForegroundColor Green
    } else {
        $config.License.ExpiresAt = (Get-Date).AddYears(1).ToString("yyyy-MM-dd")
        Write-Host "  License activated!" -ForegroundColor Green
    }
    
    Write-Host "  Key:     $Key" -ForegroundColor Cyan
    Write-Host "  Tier:    $tier" -ForegroundColor Cyan
    Write-Host "  Expires: $(if ($tierCode -eq 'P') { 'NEVER' } else { $config.License.ExpiresAt })" -ForegroundColor Cyan
    
    $config | ConvertTo-Json -Depth 10 | Out-File $configPath -Encoding UTF8 -Force
    Write-Host ""
    exit 0
}

Write-Host "  Usage:" -ForegroundColor Yellow
Write-Host "    .\Activate-License.ps1 -Key 'SENT-XXXX-XXXX-XXXX-XXXX'"
Write-Host "    .\Activate-License.ps1 -Trial"
Write-Host "    .\Activate-License.ps1 -Status"
Write-Host "    .\Activate-License.ps1 -Deactivate"
Write-Host ""
Write-Host "  Purchase: https://www.dbaops.solutions/pricing" -ForegroundColor Cyan
Write-Host "  Support:  licensing@dbaopsframework.com" -ForegroundColor Cyan
Write-Host ""
